/******************************************************************************
   Copyright (C) Roper Scientific, Inc. 1990-2000. All rights reserved.
******************************************************************************/
#ifndef _PVCAM_H
#define _PVCAM_H

/*********************** Constant & Type Definitions *************************/

/************************ Class 2: Data types ********************************/
/* Data type used by pl_get_param with attribute type (ATTR_TYPE).          */
#define TYPE_CHAR_PTR     13      /* char                                   */
#define TYPE_INT8         12      /* signed char                            */
#define TYPE_UNS8          5      /* unsigned char                          */
#define TYPE_INT16         1      /* short                                  */
#define TYPE_UNS16         6      /* unsigned short                         */
#define TYPE_INT32         2      /* long                                   */
#define TYPE_UNS32         7      /* unsigned long                          */
#define TYPE_FLT64         4      /* double                                 */
#define TYPE_ENUM          9      /* Can be treat as unsigned long          */
#define TYPE_BOOLEAN      11      /* Boolean value                          */
#define TYPE_VOID_PTR     14      /* ptr to void                            */
#define TYPE_VOID_PTR_PTR 15      /* void ptr to a ptr.                     */


/* defines for classes                                                      */
#define CLASS0      0       /* Camera Communications                        */
#define CLASS1      1       /* Error Reporting                              */
#define CLASS2      2       /* Configuration/Setup                          */
#define CLASS3      3       /* Data Acuisition                              */
#define CLASS4      4       /* Buffer Manipulation                          */
#define CLASS5      5       /* Analysis                                     */
#define CLASS6      6       /* Data Export                                  */
#define CLASS29     29      /* Buffer Functions                             */
#define CLASS30     30      /* Utility functions                            */
#define CLASS31     31      /* Memory Functions                             */
#define CLASS32     32      /* CCL Engine                                   */
#define CLASS91		91		/* RS170										*/

/************************ Parameter IDs **************************************/
/* Format: TTCCxxxx, where TT = Data type, CC = Class, xxxx = ID number      */


		/* DEVICE DRIVER PARAMETERS (CLASS 0) */

/*  Class 0 (next available index for class zero = 5) */

#define	PARAM_DD_INFO_LENGTH        ((CLASS0 << 16) + (TYPE_INT16 << 24) + 1)
#define	PARAM_DD_VERSION 			((CLASS0 << 16) + (TYPE_UNS16 << 24) + 2)
#define	PARAM_DD_RETRIES 			((CLASS0 << 16) + (TYPE_UNS16 << 24) + 3)
#define	PARAM_DD_TIMEOUT 			((CLASS0 << 16) + (TYPE_UNS16 << 24) + 4)

/* Camera Parameters Class 2 variables */

/* Class 2 (next available index for class two = 534) */

/* CCD skip parameters */
/* Min Block. amount to group on the shift register, to through way. */
#define PARAM_MIN_BLOCK             ((CLASS2 << 16) + (TYPE_INT16 << 24)  + 60)
/* number of min block groups to use before valid data. */
#define PARAM_NUM_MIN_BLOCK         ((CLASS2 << 16) + (TYPE_INT16 << 24)  + 61)
/* Strips per clear. Used to define how many clears to use for continous clears */
/* and with clears to define the clear area at the beginning of an experiment   */
#define PARAM_NUM_OF_STRIPS_PER_CLR   ((CLASS2 << 16) + (TYPE_INT16 << 24)  + 98)
/* Only applies to Thompson ST133 5Mhz  */
/* enables or disables anti-blooming.   */
#define PARAM_ANTI_BLOOMING           ((CLASS2 << 16) + (TYPE_ENUM << 24)  + 293)
/* This applies to ST133 1Mhz and 5Mhz and PentaMax V5 controllers. For the ST133 */
/* family this controls whether the BNC (not scan) is either not scan or shutter  */
/* for the PentaMax V5, this can be not scan, shutter, not ready, clearing, logic 0 */
/* logic 1, clearing, and not frame transfer image shift.                         */
/* See enum below for possible values                                             */
#define PARAM_LOGIC_OUTPUT            ((CLASS2 << 16) + (TYPE_ENUM << 24)  + 66)
/* Edge Trigger defines whether the external sync trigger is positive or negitive */
/* edge active. This is for the ST133 family (1 and 5 Mhz) and PentaMax V5.0.     */
/* see enum below for possible values.                                            */
#define PARAM_EDGE_TRIGGER            ((CLASS2 << 16) + (TYPE_ENUM << 24)  + 106)
/* Intensifier gain is currently only used by the PI-Max and has a range of 0-255 */
#define PARAM_INTENSIFIER_GAIN        ((CLASS2 << 16) + (TYPE_INT16 << 24)  + 216)
/* Shutter, Gate, or Safe mode, for the PI-Max. */
#define PARAM_SHTR_GATE_MODE           ((CLASS2 << 16) + (TYPE_ENUM << 24)  + 217)
/* ADC offset setting. */
#define	PARAM_ADC_OFFSET              ((CLASS2 << 16) + (TYPE_INT16 << 24)  + 195)
/* CCD chip name.    */
#define	PARAM_CHIP_NAME               ((CLASS2 << 16) + (TYPE_CHAR_PTR << 24)  + 129)

#define	PARAM_COOLING_MODE            ((CLASS2 << 16) + (TYPE_ENUM  << 24)  + 214)
#define	PARAM_PREAMP_DELAY            ((CLASS2 << 16) + (TYPE_UNS16 << 24)  + 502)
#define	PARAM_PREFLASH                ((CLASS2 << 16) + (TYPE_UNS16 << 24)  + 503)
#define	PARAM_COLOR_MODE              ((CLASS2 << 16) + (TYPE_ENUM  << 24)  + 504)
#define	PARAM_MPP_CAPABLE             ((CLASS2 << 16) + (TYPE_ENUM  << 24)  + 224)
#define	PARAM_PREAMP_OFF_CONTROL      ((CLASS2 << 16) + (TYPE_UNS32 << 24)  + 507)
#define	PARAM_SERIAL_NUM              ((CLASS2 << 16) + (TYPE_UNS16 << 24)  + 508)

/* CCD Dimensions and physical characteristics */
/* pre and post dummies of CCD. */
#define	PARAM_PREMASK                 ((CLASS2 << 16) + (TYPE_UNS16 << 24)  + 53)
#define	PARAM_PRESCAN                 ((CLASS2 << 16) + (TYPE_UNS16 << 24)  + 55)
#define	PARAM_POSTMASK                ((CLASS2 << 16) + (TYPE_UNS16 << 24)  + 54)
#define	PARAM_POSTSCAN                ((CLASS2 << 16) + (TYPE_UNS16 << 24)  + 56)
#define	PARAM_PIX_PAR_DIST            ((CLASS2 << 16) + (TYPE_UNS16 << 24)  + 500)
#define	PARAM_PIX_PAR_SIZE            ((CLASS2 << 16) + (TYPE_UNS16 << 24)  + 63)
#define	PARAM_PIX_SER_DIST            ((CLASS2 << 16) + (TYPE_UNS16 << 24)  + 501)
#define	PARAM_PIX_SER_SIZE            ((CLASS2 << 16) + (TYPE_UNS16 << 24)  + 62)
#define	PARAM_SUMMING_WELL            ((CLASS2 << 16) + (TYPE_BOOLEAN << 24)  + 505)
#define	PARAM_FWELL_CAPACITY          ((CLASS2 << 16) + (TYPE_UNS32 << 24)  + 506)
/* Y dimension of active area of CCD chip */
#define	PARAM_PAR_SIZE                ((CLASS2 << 16) + (TYPE_UNS16 << 24)  + 57)
/* X dimension of active area of CCD chip */
#define	PARAM_SER_SIZE                ((CLASS2 << 16) + (TYPE_UNS16 << 24)  + 58)




/* General parameters */
/* Is the controller on and running? */
#define PARAM_CONTROLLER_ALIVE      ((CLASS2 << 16) + (TYPE_BOOLEAN << 24)+ 168)
/* Readout time of current ROI, in ms */
#define PARAM_READOUT_TIME          ((CLASS2 << 16) + (TYPE_FLT64 << 24)  + 179)





		/* CAMERA PARAMETERS (CLASS 2) */

#define	PARAM_CLEAR_CYCLES      ((CLASS2 << 16) + (TYPE_UNS16 << 24)  + 97)
#define	PARAM_CLEAR_MODE        ((CLASS2 << 16) + (TYPE_ENUM << 24)  + 523)
#define PARAM_FRAME_CAPABLE 	((CLASS2 << 16) + (TYPE_BOOLEAN << 24)  + 509)
#define	PARAM_PMODE             ((CLASS2 << 16) + (TYPE_ENUM  << 24)  + 524)
#define	PARAM_CCS_STATUS 		((CLASS2 << 16) + (TYPE_INT16 << 24)  + 510)

/* This is the actual temperature of the detector. This is only a get, not a set */
#define	PARAM_TEMP				((CLASS2 << 16) + (TYPE_INT16 << 24)  + 525)
/* This is the desired temperature to set. */
#define	PARAM_TEMP_SETPOINT 	((CLASS2 << 16) + (TYPE_INT16 << 24)  + 526)
#define	PARAM_CAM_FW_VERSION 	((CLASS2 << 16) + (TYPE_UNS16 << 24)  + 532)
#define	PARAM_HEAD_SER_NUM_ALPHA ((CLASS2 << 16) + (TYPE_CHAR_PTR << 24)  + 533)
#define PARAM_PCI_FW_VERSION	((CLASS2 << 16) + (TYPE_UNS16 << 24)  + 534)

/* Exsposure mode, timed strobed etc, etc */
#define	PARAM_EXPOSURE_MODE		((CLASS2 << 16) + (TYPE_ENUM << 24)  + 535)

		/* SPEED TABLE PARAMETERS (CLASS 2) */

#define PARAM_BIT_DEPTH         ((CLASS2 << 16) + (TYPE_INT16 << 24)  + 511)
#define	PARAM_GAIN_INDEX		((CLASS2 << 16) + (TYPE_INT16 << 24)  + 512)
#define	PARAM_SPDTAB_INDEX		((CLASS2 << 16) + (TYPE_INT16 << 24)  + 513)
/* define which port (amplifier on shift register) to use. */ 
#define PARAM_READOUT_PORT      ((CLASS2 << 16) + (TYPE_ENUM << 24) + 247)
#define	PARAM_PIX_TIME			((CLASS2 << 16) + (TYPE_UNS16 << 24)  + 516)

		/* SHUTTER PARAMETERS (CLASS 2) */

#define	PARAM_SHTR_CLOSE_DELAY  ((CLASS2 << 16) + (TYPE_UNS16 << 24)  + 519)
#define	PARAM_SHTR_OPEN_DELAY   ((CLASS2 << 16) + (TYPE_UNS16 << 24)  + 520)
#define	PARAM_SHTR_OPEN_MODE    ((CLASS2 << 16) + (TYPE_ENUM  << 24)  + 521)
#define	PARAM_SHTR_STATUS       ((CLASS2 << 16) + (TYPE_ENUM  << 24)  + 522)

		/* I/O PARAMETERS (CLASS 2) */

#define	PARAM_IO_ADDR           ((CLASS2 << 16) + (TYPE_UNS16 << 24)  + 527)
#define	PARAM_IO_TYPE           ((CLASS2 << 16) + (TYPE_ENUM << 24)   + 528)
#define	PARAM_IO_DIRECTION      ((CLASS2 << 16) + (TYPE_ENUM << 24)   + 529)
#define	PARAM_IO_STATE          ((CLASS2 << 16) + (TYPE_FLT64 << 24)  + 530)
#define	PARAM_IO_BITDEPTH       ((CLASS2 << 16) + (TYPE_UNS16 << 24)  + 531)

		/* ACQUISITION PARAMETERS (CLASS 3) (next available index for class three = 5) */

#define	PARAM_EXP_TIME          ((CLASS3 << 16) + (TYPE_UNS16 << 24)  + 1)
#define	PARAM_EXP_RES           ((CLASS3 << 16) + (TYPE_ENUM << 24)   + 2)
#define	PARAM_EXP_MIN_TIME      ((CLASS3 << 16) + (TYPE_FLT64 << 24)  + 3)
#define	PARAM_EXP_RES_INDEX     ((CLASS3 << 16) + (TYPE_UNS16 << 24)  + 4)

		/* PARAMETERS FOR  BEGIN and END of FRAME Interrupts */
#define PARAM_BOF_EOF_ENABLE	((CLASS3 << 16) + (TYPE_ENUM << 24)    + 5)
#define PARAM_BOF_EOF_COUNT		((CLASS3 << 16) + (TYPE_UNS32 << 24)   + 6)
#define PARAM_BOF_EOF_CLR       ((CLASS3 << 16) + (TYPE_BOOLEAN << 24) + 7)

/* Test to see if hardware/software can perform circular buffer */
#define PARAM_CIRC_BUFFER       ((CLASS3 << 16) + (TYPE_BOOLEAN << 24) + 299)


/********************** Class 0: Abort Exposure flags ************************/
enum {  CCS_NO_CHANGE,CCS_HALT,CCS_HALT_CLOSE_SHTR,CCS_CLEAR,
        CCS_CLEAR_CLOSE_SHTR,CCS_OPEN_SHTR,CCS_CLEAR_OPEN_SHTR };

/********************** Class 0: Open Camera Modes ***************************/
enum {  OPEN_EXCLUSIVE };

/********************** Class 0: Readout status flags ************************/
enum {  READOUT_NOT_ACTIVE,EXPOSURE_IN_PROGRESS,READOUT_IN_PROGRESS,
        READOUT_COMPLETE,READOUT_FAILED,ACQUISITION_IN_PROGRESS };

/************************ Class 1: Error message size ************************/
#define ERROR_MSG_LEN  255   /* No error message will be longer than this */

/*********************** Class 2: Cooling type flags *************************/
enum {  NORMAL_COOL,CRYO_COOL };

/************************** Class 2: Name/ID sizes ***************************/
#define CCD_NAME_LEN  17          /* Includes space for the null terminator */
#define MAX_ALPHA_SER_NUM_LEN 32  /* Includes space for the null terminator */

/*********************** Class 2: MPP capability flags ***********************/
enum {  MPP_UNKNOWN,MPP_ALWAYS_OFF,MPP_ALWAYS_ON,MPP_SELECTABLE };

/************************** Class 2: Shutter flags ***************************/
enum {  SHTR_FAULT,SHTR_OPENING,SHTR_OPEN,SHTR_CLOSING,SHTR_CLOSED,
        SHTR_UNKNOWN };

/************************ Class 2: Pmode constants ***************************/
enum {  PMODE_NORMAL,PMODE_FT,PMODE_MPP,PMODE_FT_MPP,
        PMODE_ALT_NORMAL,PMODE_ALT_FT,PMODE_ALT_MPP,PMODE_ALT_FT_MPP };

/************************ Class 2: Color support constants *******************/
enum {  COLOR_NONE,COLOR_RGGB=2 };

/************************ Class 2: Attribute IDs *****************************/
enum {  ATTR_CURRENT,ATTR_COUNT,ATTR_TYPE,ATTR_MIN,ATTR_MAX,ATTR_DEFAULT,
		ATTR_INCREMENT,ATTR_ACCESS, ATTR_AVAIL };

/************************ Class 2: Access types ******************************/
enum {  ACC_ERROR,ACC_READ_ONLY,ACC_READ_WRITE,ACC_EXIST_CHECK_ONLY,
		ACC_WRITE_ONLY };
/* This enum is used by the access Attribute */

/************************ Class 2: I/O types *********************************/
enum {  IO_TYPE_TTL,IO_TYPE_DAC };

/************************ Class 2: I/O direction flags ***********************/
enum {  IO_DIR_INPUT,IO_DIR_OUTPUT,IO_DIR_INPUT_OUTPUT };

/************************ Class 2: I/O port attributes ***********************/
enum {  IO_ATTR_DIR_FIXED,IO_ATTR_DIR_VARIABLE_ALWAYS_READ };

/************************ Class 2: Trigger polarity **************************/
/* used with the PARAM_EDGE_TRIGGER parameter id.                              */
enum { EDGE_TRIG_POS=2, EDGE_TRIG_NEG };

/************************ Class 2: Logic Output ******************************/
/* used with the PARAM_LOGIC_OUTPUT parameter id.                              */
enum {OUTPUT_NOT_SCAN=0, OUTPUT_SHUTTER, OUTPUT_NOT_RDY, OUTPUT_LOGIC0,      
      OUTPUT_CLEARING, OUTPUT_NOT_FT_IMAGE_SHIFT, OUTPUT_RESERVED,          
      OUTPUT_LOGIC1};

/************************ Class 2: PI-Max intensifer gating settings *********/
/* used with the PARAM_SHTR_GATE_MODE parameter id.                             */
enum { INTENSIFIER_SAFE=0, INTENSIFIER_GATING, INTENSIFIER_SHUTTER };

/************************ Class 2: Readout Port ******************************/
/* used with the PARAM_READOUT_PORT parameter id.                            */
enum { READOUT_PORT1 /* st133 low noise */, READOUT_PORT2 /* st133 high capacity */ };

/************************ Class 2: Anti Blooming ******************************/
/* used with the PARAM_ANTI_BLOOMING parameter id.                             */
enum {ANTIBLOOM_NOTUSED = 0, ANTIBLOOM_INACTIVE, ANTIBLOOM_ACTIVE};


/************************ Class 3: Clearing mode flags ***********************/
enum {  CLEAR_NEVER,CLEAR_PRE_EXPOSURE,CLEAR_PRE_SEQUENCE,CLEAR_POST_SEQUENCE,
        CLEAR_PRE_POST_SEQUENCE,CLEAR_PRE_EXPOSURE_POST_SEQ };

/************************ Class 3: Shutter mode flags ************************/
enum {  OPEN_NEVER,OPEN_PRE_EXPOSURE,OPEN_PRE_SEQUENCE,OPEN_PRE_TRIGGER,
        OPEN_NO_CHANGE };

/************************ Class 3: Exposure mode flags ***********************/
enum {  TIMED_MODE,STROBED_MODE,BULB_MODE,TRIGGER_FIRST_MODE,FLASH_MODE,
        VARIABLE_TIMED_MODE };

/************************ Class 3: Event constants ***************************/
enum {  EVENT_START_READOUT,EVENT_END_READOUT };

/************************ Class 3: EOF/BOF constants *************************/
enum {  NO_FRAME_IRQS,BEGIN_FRAME_IRQS,END_FRAME_IRQS,BEGIN_END_FRAME_IRQS};

/************************ Class 3: Continuous Mode constants *****************/
enum {  CIRC_NONE,CIRC_OVERWRITE,CIRC_NO_OVERWRITE };

/************************ Class 3: Fast Exposure Resolution constants ********/
enum {  EXP_RES_ONE_MILLISEC,EXP_RES_ONE_MICROSEC };

/************************ Class 3: I/O Script Locations **********************/
enum {  SCR_PRE_OPEN_SHTR,SCR_POST_OPEN_SHTR,SCR_PRE_FLASH,SCR_POST_FLASH,
		SCR_PRE_INTEGRATE,SCR_POST_INTEGRATE,SCR_PRE_READOUT,SCR_POST_READOUT,
		SCR_PRE_CLOSE_SHTR,SCR_POST_CLOSE_SHTR };

/************************* Class 3: Region Definition ************************/
typedef struct {
  uns16 s1;         /* First pixel in the serial register */
  uns16 s2;         /* Last pixel in the serial register */
  uns16 sbin;       /* Serial binning for this region */
  uns16 p1;         /* First row in the parallel register */
  uns16 p2;         /* Last row in the parallel register */
  uns16 pbin;       /* Parallel binning for this region */
} rgn_type, PV_PTR_DECL rgn_ptr;
typedef const rgn_type PV_PTR_DECL rgn_const_ptr;

/********************** Class 4: Buffer bit depth flags **********************/
enum {  PRECISION_INT8,PRECISION_UNS8,PRECISION_INT16,PRECISION_UNS16,
        PRECISION_INT32,PRECISION_UNS32 };

/************************** Class 6: Export Control **************************/
typedef struct {
  boolean rotate;       /* TRUE=Rotate the data during export */
  boolean x_flip;       /* TRUE=Flip the data horizontally during export */
  boolean y_flip;       /* TRUE=Flip the data vertically during export */
  int16 precision;      /* Bits in output data, see constants */
  int16 windowing;      /* See list of constants */
  int32 max_inten;      /* Highest intensity, if windowing */
  int32 min_inten;      /* Lowest intensity, if windowing */
  int16 output_x_size;  /* Controls output array size */
  int16 output_y_size;  /* Controls output array size */
} export_ctrl_type, PV_PTR_DECL export_ctrl_ptr;
typedef const export_ctrl_type PV_PTR_DECL export_ctrl_const_ptr;

/**************************** Function Prototypes ****************************/
#ifdef PV_C_PLUS_PLUS
extern "C" {
#endif

/************* Class 0: Camera Communications Function Prototypes ************/
boolean PV_DECL pl_cam_check(int16 hcam);
boolean PV_DECL pl_cam_close(int16 hcam);
boolean PV_DECL pl_cam_get_diags(int16 hcam);
boolean PV_DECL pl_cam_get_name(int16 cam_num,char_ptr cam_name);
boolean PV_DECL pl_cam_get_total(int16_ptr totl_cams);
boolean PV_DECL pl_cam_open(char_ptr camera_name,int16_ptr hcam, int16 o_mode );

boolean PV_DECL pl_dd_get_info(int16 hcam,int16 bytes,char_ptr text);
boolean PV_DECL pl_dd_get_info_length(int16 hcam,int16_ptr bytes);
boolean PV_DECL pl_dd_get_ver(int16 hcam,uns16_ptr version);
boolean PV_DECL pl_dd_get_retries(int16 hcam,uns16_ptr max_retries);
boolean PV_DECL pl_dd_get_timeout(int16 hcam,uns16_ptr m_sec);
boolean PV_DECL pl_dd_set_retries(int16 hcam,uns16 max_retries);
boolean PV_DECL pl_dd_set_timeout(int16 hcam,uns16 m_sec);

boolean PV_DECL pl_ddi_get_ver(uns16_ptr version);

boolean PV_DECL pl_exp_abort(int16 hcam,int16 cam_state);
boolean PV_DECL pl_exp_check_cont_status(int16 hcam,int16_ptr status,
										 uns32_ptr byte_cnt,uns32_ptr buffer_cnt);
boolean PV_DECL pl_exp_check_status(int16 hcam,int16_ptr status,uns32_ptr byte_cnt);
boolean PV_DECL pl_exp_check_progress(int16 hcam,int16_ptr status,uns32_ptr byte_cnt);

boolean PV_DECL pl_subsys_do_diag(int16 hcam,uns8 subsys_id,uns16_ptr err_code);
boolean PV_DECL pl_subsys_get_id(int16 hcam,uns8 subsys_id,uns16_ptr part_num,
                                 uns8_ptr revision);
boolean PV_DECL pl_subsys_get_name(int16 hcam,uns8 subsys_id,char_ptr subsys_name);

/**************** Class 1: Error Reporting Function Prototypes ***************/
int16 PV_DECL pl_error_code(void);
boolean PV_DECL pl_error_message(int16 err_code,char_ptr msg);

/************** Class 2: Configuration/Setup Function Prototypes *************/
boolean PV_DECL pl_ccd_get_adc_offset( int16 hcam, int16_ptr offset );
boolean PV_DECL pl_ccd_get_chip_name(int16 hcam,char_ptr chip_name);
boolean PV_DECL pl_ccd_get_clear_cycles(int16 hcam,uns16_ptr clear_cycles);
boolean PV_DECL pl_ccd_get_clear_mode(int16 hcam,int16_ptr clear_mode);
boolean PV_DECL pl_ccd_get_cooling_mode(int16 hcam,int16_ptr cooling);
boolean PV_DECL pl_ccd_get_frame_capable(int16 hcam,boolean_ptr frame_capable);
boolean PV_DECL pl_ccd_get_fwell_capacity(int16 hcam,uns32_ptr fwell_capacity);
boolean PV_DECL pl_ccd_get_mpp_capable(int16 hcam,int16_ptr mpp_capable);
boolean PV_DECL pl_ccd_get_preamp_dly(int16 hcam,uns16_ptr preamp_dly);
boolean PV_DECL pl_ccd_get_preflash(int16 hcam,uns16_ptr pre_flash);
boolean PV_DECL pl_ccd_get_pmode(int16 hcam,int16_ptr pmode);
boolean PV_DECL pl_ccd_get_premask(int16 hcam,uns16_ptr pre_mask);
boolean PV_DECL pl_ccd_get_prescan(int16 hcam,uns16_ptr prescan);
boolean PV_DECL pl_ccd_get_postmask(int16 hcam,uns16_ptr post_mask);
boolean PV_DECL pl_ccd_get_postscan(int16 hcam,uns16_ptr postscan);
boolean PV_DECL pl_ccd_get_par_size(int16 hcam,uns16_ptr par_size);
boolean PV_DECL pl_ccd_get_serial_num(int16 hcam,uns16_ptr serial_num);
boolean PV_DECL pl_ccd_get_ser_size(int16 hcam,uns16_ptr ser_size);
boolean PV_DECL pl_ccs_get_status(int16 hcam,int16_ptr ccs_status);
boolean PV_DECL pl_ccd_get_summing_well(int16 hcam,boolean_ptr s_well_exists);
boolean PV_DECL pl_ccd_get_tmp(int16 hcam,int16_ptr cur_tmp);
boolean PV_DECL pl_ccd_get_tmp_range(int16 hcam,int16_ptr tmp_hi_val,int16_ptr tmp_lo_val);
boolean PV_DECL pl_ccd_get_tmp_setpoint(int16 hcam,int16_ptr tmp_setpoint);
boolean PV_DECL pl_ccd_get_pix_par_dist(int16 hcam,uns16_ptr pix_par_dist);
boolean PV_DECL pl_ccd_get_pix_par_size(int16 hcam,uns16_ptr pix_par_size);
boolean PV_DECL pl_ccd_get_pix_ser_dist(int16 hcam,uns16_ptr pix_ser_dist);
boolean PV_DECL pl_ccd_get_pix_ser_size(int16 hcam,uns16_ptr pix_ser_size);
boolean PV_DECL pl_ccd_set_adc_offset( int16 hcam, int16 offset );
boolean PV_DECL pl_ccd_set_clear_cycles(int16 hcam,uns16 clr_cycles);
boolean PV_DECL pl_ccd_set_clear_mode(int16 hcam,int16 ccd_clear);
boolean PV_DECL pl_ccd_set_pmode(int16 hcam,int16 pmode);
boolean PV_DECL pl_ccd_set_tmp_setpoint(int16 hcam,int16 tmp_setpoint);
boolean PV_DECL pl_ccd_get_color_mode(int16 hcam,uns16_ptr color_mode);
boolean PV_DECL pl_ccd_get_preamp_off_control(int16 hcam, uns32_ptr preamp_off_control);
boolean PV_DECL pl_ccd_set_preamp_off_control(int16 hcam, uns32 preamp_off_control);

boolean PV_DECL pl_spdtab_get_bits(int16 hcam,int16_ptr spdtab_bits);
boolean PV_DECL pl_spdtab_get_entries(int16 hcam,int16_ptr spdtab_entries);
boolean PV_DECL pl_spdtab_get_gain(int16 hcam,int16_ptr spdtab_gain);
boolean PV_DECL pl_spdtab_get_max_gain(int16 hcam,int16_ptr spdtab_max_gain);
boolean PV_DECL pl_spdtab_get_num(int16 hcam,int16_ptr spdtab_num);
boolean PV_DECL pl_spdtab_get_port(int16 hcam,int16_ptr spdtab_port);
boolean PV_DECL pl_spdtab_get_port_total(int16 hcam,int16_ptr total_ports);
boolean PV_DECL pl_spdtab_get_time(int16 hcam,uns16_ptr spdtab_time);
boolean PV_DECL pl_spdtab_set_gain(int16 hcam,int16 spdtab_gain);
boolean PV_DECL pl_spdtab_set_num(int16 hcam,int16 spdtab_num);

boolean PV_DECL pl_pvcam_get_ver(uns16_ptr version);
boolean PV_DECL pl_pvcam_init(void);
boolean PV_DECL pl_pvcam_uninit(void);

boolean PV_DECL pl_shtr_get_close_dly(int16 hcam,uns16_ptr shtr_close_dly);
boolean PV_DECL pl_shtr_get_open_dly(int16 hcam,uns16_ptr shtr_open_dly);
boolean PV_DECL pl_shtr_get_open_mode(int16 hcam,int16_ptr shtr_open_mode);
boolean PV_DECL pl_shtr_get_status(int16 hcam,int16_ptr shtr_status);
boolean PV_DECL pl_shtr_set_close_dly(int16 hcam,uns16 shtr_close_dly);
boolean PV_DECL pl_shtr_set_open_dly(int16 hcam,uns16 shtr_open_dly);
boolean PV_DECL pl_shtr_set_open_mode(int16 hcam,int16 shtr_open_mode);

boolean PV_DECL pl_get_param(int16 hcam, uns32 param_id, int16 param_attribute,
							 void_ptr param_value);
boolean PV_DECL pl_set_param(int16 hcam, uns32 param_id, void_ptr param_value);
boolean PV_DECL pl_get_enum_param(int16 hcam, uns32 param_id, uns32 index,
								  int32_ptr value, char_ptr desc, uns32 length);
boolean PV_DECL pl_enum_str_length(int16 hcam, uns32 param_id, uns32 index,
								  uns32_ptr length);

/*************** Class 3: Data Acquisition Function Prototypes ***************/
boolean PV_DECL pl_exp_init_seq(void);
boolean PV_DECL pl_exp_finish_seq(int16 hcam,void_ptr pixel_stream,int16 hbuf);
boolean PV_DECL pl_exp_get_driver_buffer(int16 hcam,void_ptr_ptr pixel_stream,
										 uns32_ptr byte_cnt);
boolean PV_DECL pl_exp_get_latest_frame(int16 hcam,void_ptr_ptr frame);
boolean PV_DECL pl_exp_get_oldest_frame(int16 hcam,void_ptr_ptr frame);
boolean PV_DECL pl_exp_set_cont_mode(int16 hcam,int16 mode);
boolean PV_DECL pl_exp_setup_seq(int16 hcam,uns16 exp_total,uns16 rgn_total,
                                 rgn_const_ptr rgn_array,int16 mode,uns32
                                 exposure_time,uns32_ptr exp_bytes);
boolean PV_DECL pl_exp_start_cont(int16 hcam,void_ptr pixel_stream,uns32 size);
boolean PV_DECL pl_exp_start_seq(int16 hcam,void_ptr pixel_stream);
boolean PV_DECL pl_exp_stop_cont(int16 hcam,int16 cam_state);
boolean PV_DECL pl_exp_uninit_seq(void);
boolean PV_DECL pl_exp_get_time_seq(int16 hcam,uns16_ptr exp_time);
boolean PV_DECL pl_exp_set_time_seq(int16 hcam,uns16 exp_time);
boolean PV_DECL pl_exp_unlock_oldest_frame(int16 hcam);
boolean PV_DECL pl_io_script_control(int16 hcam,uns16 addr,flt64 state,
									 uns32 location);
boolean PV_DECL pl_io_clear_script_control(int16 hcam);
boolean PV_DECL pl_exp_setup_cont(		// setup circular buffer
			int16 hcam, 				// camera handle
			uns16 rgn_total,			// number of regions of interest in image	
            rgn_const_ptr rgn_array,	// regions of interest
			int16 exp_mode, 			// exposure mode (TIMEDMODE, etc...)
			uns32  exposure_time, 		// exposure time in milliseconds/microseconds
			uns32_ptr exp_bytes, 		// image size (returned sum of regions)
			int16 buffer_mode			// circular buffer mode(CIRC_OVERWRITE, etc...)
			);
boolean PV_DECL pl_exp_wait_start_xfer(int16 hcam, uns32 tlimit);
boolean PV_DECL pl_exp_wait_end_xfer(int16 hcam, uns32 tlimit);

boolean PV_DECL pl_exp_unravel(int16 hcam,				/* Handle Of the Camera	*/
							   uns16 exposure,			/* -1 for All, otherwise exposure to unravel */
							   void_ptr pixel_stream,	/* Buffer that holds the date...(User or Driver)	*/
							   uns16 rgn_total,			/* Number Of Regions	*/
                               rgn_const_ptr rgn_array, /* Array Of Regions	*/
							   uns16_ptr *array_list);	/* Array Of Pointers, In the Same Order as regions	*/

/************* Class 4: Buffer Manipulation Function Prototypes **************/
boolean PV_DECL pl_buf_init(void);
boolean PV_DECL pl_buf_uninit(void);

boolean PV_DECL pl_buf_alloc(int16_ptr hbuf,int16 exp_total,int16 bit_depth,
                             int16 rgn_total,rgn_const_ptr rgn);
boolean PV_DECL pl_buf_get_bits(int16 hbuf,int16_ptr bit_depth);
boolean PV_DECL pl_buf_get_exp_date(int16 hbuf,int16 exp_num,int16_ptr year,
                                    uns8_ptr month,uns8_ptr day,uns8_ptr hour,
                                    uns8_ptr min,uns8_ptr sec,uns16_ptr msec);
boolean PV_DECL pl_buf_get_exp_time(int16 hbuf,int16 exp_num,uns32_ptr exp_msec);
boolean PV_DECL pl_buf_get_exp_total(int16 hbuf,int16_ptr total_exps);
boolean PV_DECL pl_buf_get_img_bin(int16 himg,int16_ptr ibin,int16_ptr jbin);
boolean PV_DECL pl_buf_get_img_handle(int16 hbuf,int16 exp_num,int16 img_num,
                                      int16_ptr himg);
boolean PV_DECL pl_buf_get_img_ofs(int16 himg,int16_ptr s_ofs,int16_ptr p_ofs);
boolean PV_DECL pl_buf_get_img_ptr(int16 himg,void_ptr_ptr img_addr);
boolean PV_DECL pl_buf_get_img_size(int16 himg,int16_ptr x_size,int16_ptr y_size);
boolean PV_DECL pl_buf_get_img_total(int16 hbuf,int16_ptr totl_imgs);
boolean PV_DECL pl_buf_get_size(int16 hbuf,int32_ptr buf_size);
boolean PV_DECL pl_buf_free(int16 hbuf);
boolean PV_DECL pl_buf_set_exp_date(int16 hbuf,int16 exp_num,int16 year,
                                    uns8 month,uns8 day,uns8 hour,uns8 min,
                                    uns8 sec,uns16 msec);

#ifdef PV_C_PLUS_PLUS
};
#endif

/************************* SINGLE-BYTE MACROS ********************************/
/* These will pull out a single uns8 from either a two-uns8 integer quantity,*/
/* or a four-uns8 integer quantity.  They ARE NOT machine specific.  The     */
/* software using them is responsible for handling the interface requirements*/
/* of the NGC camera, which expects the high uns8 first, then the lower bytes*/
/* in order.  There are several macros:                                      */
/*                                                                           */
/*   ??16_BYTE extract one of two bytes from a two-uns8 integer input        */
/*   VAL_UNS16 produces a two-uns8 integer value from high & low uns8 input  */
/*   ??32_BYTE extracts one of four bytes from a four-uns8 integer input     */
/*   VAL_UNS32 produces a four-uns8 integer value from 4 input bytes         */

#define MS16_BYTE(two_byte_value) ((uns8) ((two_byte_value) >> 8))
#define LS16_BYTE(two_byte_value) ((uns8) (two_byte_value))

#define VAL_UNS16(ms_byte,ls_byte) ( (uns16)(((uns16)((uns8)(ms_byte)) << 8) |\
                                             ((uns16)((uns8)(ls_byte)))) )

#define MS32_BYTE(four_byte_val) ((uns8) ((four_byte_val) >> 24))
#define MH32_BYTE(four_byte_val) ((uns8) ((four_byte_val) >> 16))
#define ML32_BYTE(four_byte_val) ((uns8) ((four_byte_val) >> 8))
#define LS32_BYTE(four_byte_val) ((uns8) (four_byte_val))

#define VAL_UNS32(ms_byte,mh_byte,ml_byte,ls_byte) (\
  ((uns32)((uns8)(ms_byte)) << 24) |\
  ((uns32)((uns8)(mh_byte)) << 16) |\
  ((uns32)((uns8)(ml_byte)) << 8)  |\
  ((uns32)((uns8)(ls_byte)) ) )

#endif /* _PVCAM_H */
